#!/usr/bin/Rscript
results_tex <- "../results.tex"
clear_results <- function() if(file.exists(results_tex)) . <- file.remove(results_tex)
append_result <- function(...) write(sub("\n$", "", sprintf(...)), results_tex, append=T)
append_percent <- function(...) append_result("\\def\\%s{\\qtypercent{%f}}", ...)
append_second <- function(...) append_result("\\def\\%s{\\qtysecond{%f}}", ...)
append_gibibyte <- function(...) append_result("\\def\\%s{\\qtygibibyte{%f}}", ...)

################################################################################
# CONUS
runs_conus <- read.csv("runs_conus.csv", stringsAsFactors=T)

# reorder factors
runs_conus$method <- factor(runs_conus$method,
			    levels=c("meuflm",  # more memory
				     "meufll",  # less memory
				     "meuflL")) # least memory

# rename factors
levels(runs_conus$method) <- c("MEUFLm", "MEUFLl", "MEUFLL")

# merge cells
conus_cells <- read.csv("fdr_conus_tl.cells")
conus_cells <- conus_cells$cells

runs_conus$cells <- conus_cells

methods <- levels(runs_conus$method)
threads <- sort(unique(runs_conus$threads[runs_conus$trial==1&
		       runs_conus$method==methods[1]]))
threads.max <- max(threads)
trials <- sort(unique(runs_conus$trial[runs_conus$threads==1&
		      runs_conus$method==methods[1]]))

################################################################################
# states
runs_states <- read.csv("runs_states.csv", stringsAsFactors=T)

# reorder factors
runs_states$method <- factor(runs_states$method,
			     levels=c("meuflm",  # more memory
				      "meufll",  # less memory
				      "meuflL")) # least memory

# rename factors
levels(runs_states$method) <- c("MEUFLm", "MEUFLl", "MEUFLL")

# merge cells
state_cells <- read.csv("fdr_states.cells")
runs_states <- merge(runs_states, state_cells)

states <- sort(state_cells$stusps)

################################################################################
# counties
runs_counties <- read.csv("runs_counties.csv", stringsAsFactors=T)

# reorder factors
runs_counties$method <- factor(runs_counties$method,
			       levels=c("meuflm",  # more memory
					"meufll",  # less memory
					"meuflL")) # least memory

# rename factors
levels(runs_counties$method) <- c("MEUFLm", "MEUFLl", "MEUFLL")


county_cats <- read.table("conus_countyns.cats", sep="\t", quote="")
colnames(county_cats) <- c("countyns", "county")

county_cells <- read.csv("fdr_counties.cells")

runs_counties <- merge(runs_counties, merge(county_cats, county_cells))

clear_results()

append_result("\\def\\MaxThrCount{%d}", threads.max)

################################################################################
# CONUS
################################################################################
# compute
compute_conus_by_m <- aggregate(runs_conus$compute,
				list(method=runs_conus$method), mean)
colnames(compute_conus_by_m)[ncol(compute_conus_by_m)] <- "compute"

x <- as.data.frame(t(compute_conus_by_m$compute))
colnames(x) <- methods

# overheads
append_percent("AvgOHPctComputeConusMEUFLl", (x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgOHPctComputeConusMEUFLL", (x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

compute_conus_by_tm <- aggregate(runs_conus$compute,
				 list(threads=runs_conus$threads,
				      method=runs_conus$method), mean, na.rm=T)
colnames(compute_conus_by_tm)[ncol(compute_conus_by_tm)] <- "compute"

x <- reshape(compute_conus_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- conus_cells
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../compute-conus-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrOHPctComputeConusMEUFLl", (y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrOHPctComputeConusMEUFLL", (y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

append_second("MaxThrComputeConusMEUFLm", y$MEUFLm)
append_second("MaxThrComputeConusMEUFLl", y$MEUFLl)
append_second("MaxThrComputeConusMEUFLL", y$MEUFLL)

x[,methods] <- x$cells/x[,methods]
write.csv(x, "../throughput-conus-by-threads.csv", quote=F, row.names=F, na="nan")

################################################################################
# cpu
cpu_conus_by_m <- aggregate(runs_conus$cpu,
			    list(method=runs_conus$method), mean)
colnames(cpu_conus_by_m)[ncol(cpu_conus_by_m)] <- "cpu"

x <- as.data.frame(t(cpu_conus_by_m$cpu))
colnames(x) <- methods

# overheads
append_percent("AvgOHPctCpuConusMEUFLl", (x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgOHPctCpuConusMEUFLL", (x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

cpu_conus_by_tm <- aggregate(runs_conus$cpu,
			     list(threads=runs_conus$threads,
				  method=runs_conus$method), mean, na.rm=T)
colnames(cpu_conus_by_tm)[ncol(cpu_conus_by_tm)] <- "cpu"

x <- reshape(cpu_conus_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- conus_cells
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../cpu-conus-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrOHPctCpuConusMEUFLl", (y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrOHPctCpuConusMEUFLL", (y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

################################################################################
# mempeak
mempeak_conus_by_m <- aggregate(runs_conus$mempeak,
				list(method=runs_conus$method), mean)
colnames(mempeak_conus_by_m)[ncol(mempeak_conus_by_m)] <- "mempeak"

x <- as.data.frame(t(mempeak_conus_by_m$mempeak))
colnames(x) <- methods

# overheads
append_percent("AvgSavPctMempeakConusMEUFLl", -(x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgSavPctMempeakConusMEUFLL", -(x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

mempeak_conus_by_tm <- aggregate(runs_conus$mempeak,
				 list(threads=runs_conus$threads,
				      method=runs_conus$method), mean, na.rm=T)
colnames(mempeak_conus_by_tm)[ncol(mempeak_conus_by_tm)] <- "mempeak"

x <- reshape(mempeak_conus_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- conus_cells
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../mempeak-conus-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# savings
append_percent("MaxThrSavPctMempeakConusMEUFLl", -(y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrSavPctMempeakConusMEUFLL", -(y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

append_gibibyte("MaxThrMempeakConusMEUFLm", y$MEUFLm/1024^3)
append_gibibyte("MaxThrMempeakConusMEUFLl", y$MEUFLl/1024^3)
append_gibibyte("MaxThrMempeakConusMEUFLL", y$MEUFLL/1024^3)

################################################################################
# memaccess
memaccess_conus_by_m <- aggregate(runs_conus$memaccess,
				  list(method=runs_conus$method), mean)
colnames(memaccess_conus_by_m)[ncol(memaccess_conus_by_m)] <- "memaccess"

x <- as.data.frame(t(memaccess_conus_by_m$memaccess))
colnames(x) <- methods

# overheads
append_percent("AvgSavPctMemaccessConusMEUFLl", -(x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgSavPctMemaccessConusMEUFLL", -(x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

memaccess_conus_by_tm <- aggregate(runs_conus$memaccess,
				   list(threads=runs_conus$threads,
					method=runs_conus$method), mean, na.rm=T)
colnames(memaccess_conus_by_tm)[ncol(memaccess_conus_by_tm)] <- "memaccess"

x <- reshape(memaccess_conus_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- conus_cells
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../memaccess-conus-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# savings
append_percent("MaxThrSavPctMemaccessConusMEUFLl", -(y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrSavPctMemaccessConusMEUFLL", -(y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

################################################################################
# States
################################################################################
# compute
compute_states_by_m <- aggregate(runs_states$compute,
				 list(method=runs_states$method), mean)
colnames(compute_states_by_m)[ncol(compute_states_by_m)] <- "compute"

x <- as.data.frame(t(compute_states_by_m$compute))
colnames(x) <- methods

# overheads
append_percent("AvgOHPctComputeStatesMEUFLl", (x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgOHPctComputeStatesMEUFLL", (x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

compute_states_by_sm <- aggregate(runs_states$compute,
				  list(stusps=runs_states$stusps,
				       method=runs_states$method), mean)
colnames(compute_states_by_sm)[ncol(compute_states_by_sm)] <- "compute"

x <- reshape(compute_states_by_sm, idvar="stusps", timevar="method", direction="wide")
colnames(x) <- c("stusps", methods)
x <- merge(x, state_cells)
x <- x[order(x$stusps), c("stusps", "cells", methods)]
write.csv(x, "../compute-states-by-stusps.csv", quote=F, row.names=F, na="nan")

x[,methods] <- x$cells/x[,methods]
write.csv(x, "../throughput-states-by-stusps.csv", quote=F, row.names=F, na="nan")

compute_states_by_tm <- aggregate(runs_states$compute,
				  list(threads=runs_states$threads,
				       method=runs_states$method), mean)
colnames(compute_states_by_tm)[ncol(compute_states_by_tm)] <- "compute"

x <- reshape(compute_states_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(state_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../compute-states-by-threads.csv", quote=F, row.names=F, na="nan")

x[,methods] <- x$cells/x[,methods]
write.csv(x, "../throughput-states-by-threads.csv", quote=F, row.names=F, na="nan")

compute_states_by_stm <- aggregate(runs_states$compute,
				   list(stusps=runs_states$stusps,
					threads=runs_states$threads,
					method=runs_states$method), mean, na.rm=T)
colnames(compute_states_by_stm)[ncol(compute_states_by_stm)] <- "compute"

x <- reshape(compute_states_by_stm, idvar=c("stusps", "threads"), timevar="method", direction="wide")
colnames(x) <- c("stusps", "threads", methods)
x <- merge(x, state_cells)
x <- x[order(x$stusps, x$threads), c("stusps", "cells", "threads", methods)]
write.csv(x, "../compute-states-by-stusps-threads.csv", quote=F, row.names=F, na="nan")

x[,methods] <- x$cells/x[,methods]
write.csv(x, "../throughput-states-by-stusps-threads.csv", quote=F, row.names=F, na="nan")

################################################################################
# cpu
cpu_states_by_m <- aggregate(runs_states$cpu,
			     list(method=runs_states$method), mean)
colnames(cpu_states_by_m)[ncol(cpu_states_by_m)] <- "cpu"

x <- as.data.frame(t(cpu_states_by_m$cpu))
colnames(x) <- methods

# overheads
append_percent("AvgOHPctCpuStatesMEUFLl", (x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgOHPctCpuStatesMEUFLL", (x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

cpu_states_by_sm <- aggregate(runs_states$cpu,
			      list(stusps=runs_states$stusps,
				   method=runs_states$method), mean, na.rm=T)
colnames(cpu_states_by_sm)[ncol(cpu_states_by_sm)] <- "cpu"

x <- reshape(cpu_states_by_sm, idvar="stusps", timevar="method", direction="wide")
colnames(x) <- c("stusps", methods)
x <- merge(x, state_cells)
x <- x[order(x$stusps), c("stusps", "cells", methods)]
write.csv(x, "../cpu-states-by-stusps.csv", quote=F, row.names=F, na="nan")

cpu_states_by_tm <- aggregate(runs_states$cpu,
			      list(threads=runs_states$threads,
				   method=runs_states$method), mean, na.rm=T)
colnames(cpu_states_by_tm)[ncol(cpu_states_by_tm)] <- "cpu"

x <- reshape(cpu_states_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(state_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../cpu-states-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrOHPctCpuStatesMEUFLl", (y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrOHPctCpuStatesMEUFLL", (y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

################################################################################
# mempeak
mempeak_states_by_m <- aggregate(runs_states$mempeak,
				 list(method=runs_states$method), mean)
colnames(mempeak_states_by_m)[ncol(mempeak_states_by_m)] <- "mempeak"

x <- as.data.frame(t(mempeak_states_by_m$mempeak))
colnames(x) <- methods

# overheads
append_percent("AvgSavPctMempeakStatesMEUFLl", -(x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgSavPctMempeakStatesMEUFLL", -(x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

mempeak_states_by_sm <- aggregate(runs_states$mempeak,
				  list(stusps=runs_states$stusps,
				       method=runs_states$method), mean, na.rm=T)
colnames(mempeak_states_by_sm)[ncol(mempeak_states_by_sm)] <- "mempeak"

x <- reshape(mempeak_states_by_sm, idvar="stusps", timevar="method", direction="wide")
colnames(x) <- c("stusps", methods)
x <- merge(x, state_cells)
x <- x[order(x$stusps), c("stusps", "cells", methods)]
write.csv(x, "../mempeak-states-by-stusps.csv", quote=F, row.names=F, na="nan")

mempeak_states_by_tm <- aggregate(runs_states$mempeak,
				  list(threads=runs_states$threads,
				       method=runs_states$method), mean, na.rm=T)
colnames(mempeak_states_by_tm)[ncol(mempeak_states_by_tm)] <- "mempeak"

x <- reshape(mempeak_states_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(state_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../mempeak-states-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrSavPctMempeakStatesMEUFLl", -(y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrSavPctMempeakStatesMEUFLL", -(y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

################################################################################
# memaccess
memaccess_states_by_m <- aggregate(runs_states$memaccess,
				   list(method=runs_states$method), mean)
colnames(memaccess_states_by_m)[ncol(memaccess_states_by_m)] <- "memaccess"

x <- as.data.frame(t(memaccess_states_by_m$memaccess))
colnames(x) <- methods

# overheads
append_percent("AvgSavPctMemaccessStatesMEUFLl", -(x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgSavPctMemaccessStatesMEUFLL", -(x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

memaccess_states_by_sm <- aggregate(runs_states$memaccess,
				    list(stusps=runs_states$stusps,
					 method=runs_states$method), mean, na.rm=T)
colnames(memaccess_states_by_sm)[ncol(memaccess_states_by_sm)] <- "memaccess"

x <- reshape(memaccess_states_by_sm, idvar="stusps", timevar="method", direction="wide")
colnames(x) <- c("stusps", methods)
x <- merge(x, state_cells)
x <- x[order(x$stusps), c("stusps", "cells", methods)]
write.csv(x, "../memaccess-states-by-stusps.csv", quote=F, row.names=F, na="nan")

memaccess_states_by_tm <- aggregate(runs_states$memaccess,
				    list(threads=runs_states$threads,
					 method=runs_states$method), mean, na.rm=T)
colnames(memaccess_states_by_tm)[ncol(memaccess_states_by_tm)] <- "memaccess"

x <- reshape(memaccess_states_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(state_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../memaccess-states-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrSavPctMemaccessStatesMEUFLl", -(y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrSavPctMemaccessStatesMEUFLL", -(y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

################################################################################
# Counties
################################################################################
# compute
compute_counties_by_m <- aggregate(runs_counties$compute,
				   list(method=runs_counties$method), mean)
colnames(compute_counties_by_m)[ncol(compute_counties_by_m)] <- "compute"

x <- as.data.frame(t(compute_counties_by_m$compute))
colnames(x) <- methods

# overheads
append_percent("AvgOHPctComputeCountiesMEUFLl", (x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgOHPctComputeCountiesMEUFLL", (x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

compute_counties_by_sm <- aggregate(runs_counties$compute,
				    list(countyns=runs_counties$countyns,
					 method=runs_counties$method), mean)
colnames(compute_counties_by_sm)[ncol(compute_counties_by_sm)] <- "compute"

x <- reshape(compute_counties_by_sm, idvar="countyns", timevar="method", direction="wide")
colnames(x) <- c("countyns", methods)
x <- merge(x, county_cells)
x <- x[order(x$countyns), c("countyns", "cells", methods)]
write.csv(x, "../compute-counties-by-countyns.csv", quote=F, row.names=F, na="nan")

x[,methods] <- x$cells/x[,methods]
write.csv(x, "../throughput-counties-by-countyns.csv", quote=F, row.names=F, na="nan")

compute_counties_by_tm <- aggregate(runs_counties$compute,
				    list(threads=runs_counties$threads,
					 method=runs_counties$method), mean)
colnames(compute_counties_by_tm)[ncol(compute_counties_by_tm)] <- "compute"

x <- reshape(compute_counties_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(county_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../compute-counties-by-threads.csv", quote=F, row.names=F, na="nan")

x[,methods] <- x$cells/x[,methods]
write.csv(x, "../throughput-counties-by-threads.csv", quote=F, row.names=F, na="nan")

compute_counties_by_stm <- aggregate(runs_counties$compute,
				     list(countyns=runs_counties$countyns,
					  threads=runs_counties$threads,
					  method=runs_counties$method), mean, na.rm=T)
colnames(compute_counties_by_stm)[ncol(compute_counties_by_stm)] <- "compute"

x <- reshape(compute_counties_by_stm, idvar=c("countyns", "threads"), timevar="method", direction="wide")
colnames(x) <- c("countyns", "threads", methods)
x <- merge(x, county_cells)
x <- x[order(x$countyns, x$threads), c("countyns", "cells", "threads", methods)]
write.csv(x, "../compute-counties-by-countyns-threads.csv", quote=F, row.names=F, na="nan")

x[,methods] <- x$cells/x[,methods]
write.csv(x, "../throughput-counties-by-countyns-threads.csv", quote=F, row.names=F, na="nan")

################################################################################
# cpu
cpu_counties_by_m <- aggregate(runs_counties$cpu,
			     list(method=runs_counties$method), mean)
colnames(cpu_counties_by_m)[ncol(cpu_counties_by_m)] <- "cpu"

x <- as.data.frame(t(cpu_counties_by_m$cpu))
colnames(x) <- methods

# overheads
append_percent("AvgOHPctCpuCountiesMEUFLl", (x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgOHPctCpuCountiesMEUFLL", (x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

cpu_counties_by_sm <- aggregate(runs_counties$cpu,
			      list(countyns=runs_counties$countyns,
				   method=runs_counties$method), mean, na.rm=T)
colnames(cpu_counties_by_sm)[ncol(cpu_counties_by_sm)] <- "cpu"

x <- reshape(cpu_counties_by_sm, idvar="countyns", timevar="method", direction="wide")
colnames(x) <- c("countyns", methods)
x <- merge(x, county_cells)
x <- x[order(x$countyns), c("countyns", "cells", methods)]
write.csv(x, "../cpu-counties-by-countyns.csv", quote=F, row.names=F, na="nan")

cpu_counties_by_tm <- aggregate(runs_counties$cpu,
			      list(threads=runs_counties$threads,
				   method=runs_counties$method), mean, na.rm=T)
colnames(cpu_counties_by_tm)[ncol(cpu_counties_by_tm)] <- "cpu"

x <- reshape(cpu_counties_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(county_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../cpu-counties-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrOHPctCpuCountiesMEUFLl", (y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrOHPctCpuCountiesMEUFLL", (y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

################################################################################
# mempeak
mempeak_counties_by_m <- aggregate(runs_counties$mempeak,
				   list(method=runs_counties$method), mean)
colnames(mempeak_counties_by_m)[ncol(mempeak_counties_by_m)] <- "mempeak"

x <- as.data.frame(t(mempeak_counties_by_m$mempeak))
colnames(x) <- methods

# overheads
append_percent("AvgSavPctMempeakCountiesMEUFLl", -(x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgSavPctMempeakCountiesMEUFLL", -(x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

mempeak_counties_by_sm <- aggregate(runs_counties$mempeak,
				    list(countyns=runs_counties$countyns,
					 method=runs_counties$method), mean, na.rm=T)
colnames(mempeak_counties_by_sm)[ncol(mempeak_counties_by_sm)] <- "mempeak"

x <- reshape(mempeak_counties_by_sm, idvar="countyns", timevar="method", direction="wide")
colnames(x) <- c("countyns", methods)
x <- merge(x, county_cells)
x <- x[order(x$countyns), c("countyns", "cells", methods)]
write.csv(x, "../mempeak-counties-by-countyns.csv", quote=F, row.names=F, na="nan")

mempeak_counties_by_tm <- aggregate(runs_counties$mempeak,
				    list(threads=runs_counties$threads,
					 method=runs_counties$method), mean, na.rm=T)
colnames(mempeak_counties_by_tm)[ncol(mempeak_counties_by_tm)] <- "mempeak"

x <- reshape(mempeak_counties_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(county_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../mempeak-counties-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrSavPctMempeakCountiesMEUFLl", -(y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrSavPctMempeakCountiesMEUFLL", -(y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)

################################################################################
# memaccess
memaccess_counties_by_m <- aggregate(runs_counties$memaccess,
				     list(method=runs_counties$method), mean)
colnames(memaccess_counties_by_m)[ncol(memaccess_counties_by_m)] <- "memaccess"

x <- as.data.frame(t(memaccess_counties_by_m$memaccess))
colnames(x) <- methods

# overheads
append_percent("AvgSavPctMemaccessCountiesMEUFLl", -(x$MEUFLl-x$MEUFLm)/x$MEUFLm*100)
append_percent("AvgSavPctMemaccessCountiesMEUFLL", -(x$MEUFLL-x$MEUFLm)/x$MEUFLm*100)

memaccess_counties_by_sm <- aggregate(runs_counties$memaccess,
				      list(countyns=runs_counties$countyns,
					   method=runs_counties$method), mean, na.rm=T)
colnames(memaccess_counties_by_sm)[ncol(memaccess_counties_by_sm)] <- "memaccess"

x <- reshape(memaccess_counties_by_sm, idvar="countyns", timevar="method", direction="wide")
colnames(x) <- c("countyns", methods)
x <- merge(x, county_cells)
x <- x[order(x$countyns), c("countyns", "cells", methods)]
write.csv(x, "../memaccess-counties-by-countyns.csv", quote=F, row.names=F, na="nan")

memaccess_counties_by_tm <- aggregate(runs_counties$memaccess,
				      list(threads=runs_counties$threads,
					   method=runs_counties$method), mean, na.rm=T)
colnames(memaccess_counties_by_tm)[ncol(memaccess_counties_by_tm)] <- "memaccess"

x <- reshape(memaccess_counties_by_tm, idvar="threads", timevar="method", direction="wide")
colnames(x) <- c("threads", methods)
x$cells <- as.integer(mean(county_cells$cells)+0.5)
x <- x[, c("cells", "threads", methods)]
write.csv(x, "../memaccess-counties-by-threads.csv", quote=F, row.names=F, na="nan")

y <- subset(x, threads==threads.max, c("MEUFLm", "MEUFLl", "MEUFLL"))
# overheads
append_percent("MaxThrSavPctMemaccessCountiesMEUFLl", -(y$MEUFLl-y$MEUFLm)/y$MEUFLm*100)
append_percent("MaxThrSavPctMemaccessCountiesMEUFLL", -(y$MEUFLL-y$MEUFLm)/y$MEUFLm*100)
